import React, { useState } from "react";
import axios from "axios";
import MovieCard from "../components/MovieCard";

export default function Home() {
  const [title, setTitle] = useState("");
  const [movies, setMovies] = useState([]);
  const [loading, setLoading] = useState(false);

  const search = async () => {
    if (!title) return;
    setLoading(true);
    try {
      const res = await axios.get(
        `http://localhost:5000/api/movies/search?title=${encodeURIComponent(
          title
        )}`
      );
      setMovies(res.data.Search || []);
    } catch (err) {
      console.error(err);
      alert(
        "Search failed. Make sure backend is running and .env has OMDB_API_KEY."
      );
    } finally {
      setLoading(false);
    }
  };

  const handleKeyPress = (e) => {
    if (e.key === "Enter") {
      search();
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-900 via-blue-900 to-indigo-900">
      {/* Header Section */}
      <div className="container mx-auto px-6 py-12">
        <div className="text-center mb-12">
          <h1 className="text-6xl font-bold text-white mb-4 bg-gradient-to-r from-pink-400 to-purple-400 bg-clip-text text-transparent">
            Movie Finder
          </h1>
          <p className="text-gray-300 text-lg">
            Discover your next favorite movie
          </p>
        </div>

        {/* Search Bar */}
        <div className="max-w-3xl mx-auto mb-12">
          <div className="flex gap-3 bg-white bg-opacity-10 backdrop-blur-lg rounded-2xl p-2 shadow-2xl border border-white border-opacity-20">
            <input
              value={title}
              onChange={(e) => setTitle(e.target.value)}
              onKeyPress={handleKeyPress}
              placeholder="Search for movies, series, episodes..."
              className="flex-1 bg-transparent text-black placeholder-gray-300 px-6 py-4 outline-none text-lg"
            />
            <button
              onClick={search}
              disabled={loading}
              className="px-8 py-4 bg-gradient-to-r from-pink-500 to-purple-600 text-white font-semibold rounded-xl hover:from-pink-600 hover:to-purple-700 transition-all duration-300 transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg"
            >
              {loading ? "Searching..." : "Search"}
            </button>
          </div>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-20">
            <div className="animate-spin rounded-full h-16 w-16 border-t-4 border-b-4 border-pink-500"></div>
          </div>
        )}

        {/* Results Section */}
        {!loading && movies.length > 0 && (
          <div>
            <h2 className="text-2xl font-semibold text-white mb-6">
              Found {movies.length} results
            </h2>
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-6">
              {movies.map((m) => (
                <MovieCard key={m.imdbID} movie={m} />
              ))}
            </div>
          </div>
        )}

        {/* No Results */}
        {!loading && movies.length === 0 && title && (
          <div className="text-center py-20">
            <p className="text-gray-300 text-xl">
              No movies found. Try a different search term.
            </p>
          </div>
        )}

        {/* Initial State */}
        {!loading && movies.length === 0 && !title && (
          <div className="text-center py-20">
            <p className="text-gray-300 text-xl">
              Start searching to discover amazing movies!
            </p>
          </div>
        )}
      </div>
    </div>
  );
}
